
/*
 *  WAVplayer - An X11-application for playing WAV audio files
 *
 *  Copyright (c) 2000 Michael Mess (michael@michaelmess.de)
 *  All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as 
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 *  Contact the Author:
 *  Michael Mess, Hainbuchenweg 21, 49808 Lingen (Ems), Germany
 *  http://www.michaelmess.de     email: michael@michaelmess.de
 *
 * ---------------------------------------------------------------------------
 *
 *  For compiling you need to have the Qt-library installed.
 *  You can get this library at http://www.trolltech.com.
 *  You also need the Qt moc ("Meta Object Compiler") for compiling.
 *
 * ---------------------------------------------------------------------------
 *
 */

#include <qapplication.h>
#include <qpushbutton.h>
#include <qlcdnumber.h>
#include <qslider.h>
#include <qfont.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qtooltip.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <linux/soundcard.h>


#define VERSIONSTRING "Version 0.2" 

/* Blocksize fr Audio-Daten */
#define BLOCKSIZE 4096*4
/* Anzahl Blcke bei Audio-Suchlauf */
#define STEP 20

void warn(char * message)
{
  fprintf(stderr, "%s\n", message);
}

/* Diese Funktion ist notwendig, weil die Qt-Entwickler dummerweise 
 * namensgleich eine close() Funktion in QTWidget implementiert haben. 
 * (Htte man die nicht anders nennen knnen ?!? - Vorschlag: QTClose() )
 * Auerhalb von QTWidget ist close() zum Fileschlieen verfgbar.
 * Mit closefile kann man dann Files auch innerhalb von QTWidget schlieen. 
 * Ansonsten macht close() innerhalb von QTWidget einen Segmentation fault. :-(
 */
int closefile(int fd)  
{
  close(fd);
}

void SetLCD(QLCDNumber * lcd, int wert)
{
  char string[4];
  string[4]=0;
  snprintf(string, 3, "%02d", wert);
  lcd->display(string);  
}

void ClrLCD(QLCDNumber * lcd)
{
  lcd->display("  ");
}

void SetTime(QLCDNumber * mlcd, QLCDNumber * slcd, int sec)
{
  int minutes;
  int seconds;

  sec=abs(sec);
  minutes=sec/60;
  seconds=sec%60;
  if (minutes<=99) {
    SetLCD(mlcd, minutes);
    SetLCD(slcd, seconds); 
  }
  else {
    mlcd->display("--");
    slcd->display("--");
  }
}

class MyWidget : public QWidget
{
  Q_OBJECT
private:
  int playmode;
  int suchmode;
  int tstat;
  int pausemode;
  int timemode;

  int autoplay;

  int track;
  int position;
  int lasttrack;
  int disclength;

  int argc;
  char ** argv;
  int trackfile[99];
  int tracklength[99];
  int offset[99];
  int soundfile;
  char buffer[BLOCKSIZE];

  int sec;
  int minutes;
  int seconds;
  
  QLCDNumber *lcd;
  QLCDNumber *mlcd;
  QLCDNumber *slcd;
  //  CDPlayer * cdplayer;
  QPushButton *svor;
  QPushButton *srueck;
  QPushButton *tvor;
  QPushButton *trueck;
  QPushButton *playbut;
  QPushButton *stopbut;
  QPushButton *pausebut;
  QPushButton *timebut;
  QLabel *mlabel;
  QLabel *slabel;
  QLabel *playlabel;
  QLabel *timelabel;
  
  QTimer *timer;
  
  QSlider * slider;
  
public:
  MyWidget( QWidget *parent=0, const char *name=0 );
  void commandlineparameter(int argcc, char ** argvv);
  
public slots:
void tsuchlauf(int wert);
  void suchlauf(int wert);
  void beep();
  void sv1();
  void sr1();
  void sv0();
  void sr0();
  void tv();
  void tr();
  void play();
  void stop();
  void pause();
  void changetimemode();
  int dsp();
  void timeout();
};

void MyWidget::tsuchlauf(int wert)
{
  this->resize(300, 200);
  //  track+=wert;
  if (wert>0)
    if (track<lasttrack-1) 
      track++;
    else 
      return;
  
  if (wert<0 && position < 8*44100)
    track--;
  
  if (track<0) track=0;
  if (track>=lasttrack) track=lasttrack-1;
  
  if(!playmode) 
    tstat=track+1;
  SetLCD(lcd, (track+1));
  ClrLCD(mlcd);
  ClrLCD(slcd);
  position=0;
  lseek(trackfile[track], 44, SEEK_SET);
}

void MyWidget::commandlineparameter(int argcc, char ** argvv)
{
  argc=argcc;
  argv=argvv;
  lasttrack=0;
  int parzaehler=0;
  char * filename;

  while(++parzaehler<argc)
    {
      filename=argv[parzaehler];
      
      int file;
      
      if (strcasecmp(filename, "-p")==0) {
	autoplay=1;
	continue;
      }

      file=open(filename, O_RDONLY);
      
      if (file <= 0)
        {
	  fprintf(stderr, "%s: %s: ", argv[0], filename);
          perror("");
	  continue;
        }
      
      struct stat fileprop;
      
      lstat(filename, &fileprop);
      int bytes;
      bytes=fileprop.st_size;  
      
      int status;
      char header[44];
      status=read(file,header,44);
      if (status!=44)
        {
          perror("Error reading WAV-Header");
          continue;
        } 
      /* Hier soll der WAV-Header getestet werden.
       * Denn es sollen nur WAV-Dateien als Tracks abgespielt werden.
       * Alles andere hrt sich nmlich scheulich an...
       * 
       */

      /* WAVE Header testen */
      if (strncmp(header, "RIFF", 4)) {
       	fprintf(stderr, "%s is no WAVE file; RIFF-Header not found\n", filename);
	continue;
      }

      if (strncmp((header+8), "WAVE", 4)) {
       	fprintf(stderr, "%s is no WAVE file; WAVE-Header not found\n", filename);
	continue;
      }

      int arg;
      /* 16 Bit ? */
      arg = *(int*) (header+34) & 0xffff; 
      if (arg != 16) {
	fprintf(stderr, "%s has %d bit; 16 bit are required\n", filename, arg);
	continue;
      }

      /* stereo ? */
      arg = *(int*) (header+22) & 0xffff;
      if (arg != 2) {
	fprintf(stderr, "%s has %d channels; stereo (2channel) is required\n", filename, arg);
	continue;
      }

      /* 44,1 kHz ? */
      arg = *(int*) (header+24) & 0xffff;
      if (arg != 44100) {
	fprintf(stderr, "%s has %d Hz sampling rate; 44100 Hz is required\n", filename, arg);
	continue;
      }

      /* In unser Widget bernehmen */
      trackfile[lasttrack]=file;
      offset[lasttrack]=disclength;
      tracklength[lasttrack++]=bytes;
      disclength+=bytes;
    }
  /* Jetzt sind alle Parameterfiles geffnet, gecheckt und in das Track-Array
   * eingeliefert.
   */      

  if (lasttrack==0) {
    fprintf(stderr, "\nWAVplayer, %s (c) 2000 by Michael Me\nUsage: %s file...\n plays the WAV files given.\n file must be a WAVE audio file, 44.1 kHz 16 bit linear PCM stereo.\n", VERSIONSTRING, argv[0]);
  }
  
  if (autoplay) 
    play();

}

int MyWidget::dsp()
{

  if (playmode) {

    if (!pausemode) {
      /* Ein Sample in die Soundkarte schreiben */
      int bytes;
      bytes=read(trackfile[track], buffer, BLOCKSIZE);
      write(soundfile, buffer, bytes);
      
      /* Trackende testen */
      
      if (bytes<BLOCKSIZE) {
	if (track<lasttrack-1) {
	  track++;
	  position=0;
	  lseek(trackfile[track], 44, SEEK_SET);
	}     
	else 
	  stop();
	
	return 0;
      }
      position+=bytes;
    }    

    if (suchmode) {
      int skip=STEP*BLOCKSIZE*(suchmode>0 ? 1 : -1);
      if ((position+skip<tracklength[track])&&(position+skip>=0)) {
	lseek(trackfile[track], skip, SEEK_CUR);
	position+=skip;
      }
      else {
	if (position+skip<0) {
	  position=0; 
	  if (track>0) {
	    track--;
	    position=((tracklength[track]-44)/4)*4+skip;
	  }
	  lseek(trackfile[track], 44+position, SEEK_SET);
	}
	else {
	  if (track<lasttrack-1) {
	    track++;
	    position=0;
	    lseek(trackfile[track], 44+position, SEEK_SET);
	  }
	  else {
	    skip=((tracklength[track]-position)/4)*4;
	    position+=skip;
	    lseek(trackfile[track], skip, SEEK_CUR);
	  }
	}
      }
    }    
    
    SetLCD(lcd, (track+1));
    
    sec=position/(4*44100);
    if (timemode==1) /* Single remaining */
      sec=(tracklength[track]-position)/(4*44100);
    if (timemode==2) /* Total elapsed */
      sec=(position+offset[track])/(4*44100);
    if (timemode==3) /* Total remaining */
      sec= (disclength-(position+offset[track]))/(4*44100);
    SetTime(mlcd, slcd, sec);
    return 0;
  }
  else {
    if (tstat) {
      ClrLCD(mlcd);
      ClrLCD(slcd);
      SetLCD(lcd, tstat);
    }
    else {
      SetLCD(lcd, (lasttrack));
      sec=disclength/(4*44100);
      SetTime(mlcd, slcd, sec);
    }
  }
}

void MyWidget::timeout()
{
  dsp();
}

void MyWidget::changetimemode()
{
  if (playmode) {
    timemode=(timemode+1)%4;
    const char * timetext[]={"     ", "single remaining", "total", "total remaining"};
    timelabel->setText(timetext[timemode]);
  }
}

void MyWidget::suchlauf(int wert)
{
  suchmode=wert;

  this->resize(300, 200);  
}

void MyWidget::beep()
{
  fprintf(stderr, "***beeep!!!***\a\n");
}

void MyWidget::sv1()
{
  suchlauf(1);
}

void MyWidget::sr1()
{
  suchlauf(-1);
}

void MyWidget::sv0()
{
  suchlauf(0);
}

void MyWidget::sr0()
{
  suchlauf(0);
}

void MyWidget::tv()
{
  tsuchlauf(1);
}

void MyWidget::tr()
{
  tsuchlauf(-1);
}

void MyWidget::play()
{
  if (pausemode)
    playlabel->setText("Play");
  pausemode=0;
  tstat=0;
  if (!playmode && lasttrack) {
    playmode=1;
    playlabel->setText("Play");
    SetLCD(lcd, (track+1));
    ClrLCD(mlcd);
    ClrLCD(slcd);
    soundfile = open("/dev/dsp1", O_RDWR);
    if (soundfile < 0 ) {
      perror("/dev/dsp nicht verfuegbar!");
      exit(1);
    }
    ioctl(soundfile, SNDCTL_DSP_SYNC, 0);
    ioctl(soundfile, SNDCTL_DSP_RESET, 0); 
    int Samplingbits=16;
    int status;
    status = ioctl(soundfile, SOUND_PCM_WRITE_BITS, &Samplingbits);
    int Stereo=2;
    status = ioctl(soundfile,SOUND_PCM_WRITE_CHANNELS, &Stereo); 
    int Samplingrate=44100;
    status=ioctl(soundfile,SOUND_PCM_WRITE_RATE, &Samplingrate);

    int kkk=12;
    ioctl(soundfile, SNDCTL_DSP_SETFRAGMENT, &kkk);
  }
  this->resize(300, 200);
}

void MyWidget::stop()
{
  tstat=0;
  timemode=0;
  if (playmode)
    closefile(soundfile);
  playmode=0;
  playlabel->setText("    ");
  timelabel->setText("    ");
  track=0;
  position=0;
  lseek(trackfile[track], 44, SEEK_SET);
  this->resize(300, 200);
}

void MyWidget::pause()
{
  if(lasttrack) {
    if (!playmode)
      play();
    pausemode=!pausemode;
    
    playlabel->setText(pausemode ? "Pause" : "Play");
  }
}


MyWidget::MyWidget( QWidget *parent, const char *name )
  : QWidget( parent, "mein schoener name" ) 
{
  
  track=0;
  position=0;
  playmode=0;
  soundfile=0;
  suchmode=0;
  lasttrack=0;
  disclength=0;
  tstat=0;
  pausemode=0;
  timemode=0;
  autoplay=0;

  //  cdplayer = new CDPlayer();
  //  MyWidget.resize(Horizontal,Vertikal);
#define lcdsizey 100
#define butleisty 130
#define butspace 70
#define butsizex butspace*9/10
#define butsizexx butspace+butsizex
#define butsizey butsizex*2/4
#define butx 10
#define butleistyy butleisty+butspace*2/4

  svor   = new QPushButton( ">>", this, "svor" );
  svor->resize(butsizex, butsizey );
  svor->move(butx+butspace*3, butleistyy);
  svor->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(svor, SIGNAL(pressed()), this, SLOT(sv1()) );
  connect(svor, SIGNAL(released()), this, SLOT(sv0()) );
  svor->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));
  QToolTip::add(svor, "Search forward in track");

  srueck = new QPushButton( "<<", this, "srueck" );
  srueck->resize(butsizex, butsizey );
  srueck->move(butx+butspace*2, butleistyy);
  srueck->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(srueck, SIGNAL(pressed()), this, SLOT(sr1()) );
  connect(srueck, SIGNAL(released()), this, SLOT(sr0()) );
  srueck->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));
  QToolTip::add(srueck, "Search backwards in track");

  tvor    = new QPushButton( ">>|", this, "tvor" );
  tvor->resize(butsizex, butsizey );
  tvor->move(butx+butspace*1, butleistyy);
  tvor->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(tvor, SIGNAL(clicked()), this, SLOT(tv()) );
  tvor->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));
  QToolTip::add(tvor, "Next track");

  trueck  = new QPushButton( "|<<", this, "trueck" );
  trueck->resize(butsizex, butsizey );
  trueck->move(butx+butspace*0, butleistyy);
  trueck->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(trueck, SIGNAL(clicked()), this, SLOT(tr()) );
  trueck->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));
  QToolTip::add(trueck, "Previous track");

  playbut  = new QPushButton( ">", this, "Play" );
  playbut->resize(butsizexx, butsizey );
  playbut->move(butx+butspace*0, butleisty);
  playbut->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(playbut, SIGNAL(clicked()), this, SLOT(play()) );
  playbut->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));
  QToolTip::add(playbut, "Play");

  stopbut  = new QPushButton( "[]", this, "Stop" );
  stopbut->resize(butsizex, butsizey );
  stopbut->move(butx+butspace*2, butleisty);
  stopbut->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(stopbut, SIGNAL(clicked()), this, SLOT(stop()) );
  stopbut->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));  
  QToolTip::add(stopbut, "Stop");

  pausebut  = new QPushButton( "||", this, "Pause" );
  pausebut->resize(butsizex, butsizey );
  pausebut->move(butx+butspace*3, butleisty);
  pausebut->setFont( QFont( "Times", 18, QFont::Bold ) );
  connect(pausebut, SIGNAL(clicked()), this, SLOT(pause()) );
  pausebut->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));  
  QToolTip::add(pausebut, "Pause");

  timebut  = new QPushButton( "Time", this, "Timebut" );
  timebut->resize(butsizex, butsizey*3/4 );
  timebut->move(butx+butspace*3, butleisty-butsizey);
  timebut->setFont( QFont( "Times", 14, QFont::Bold ) );
  connect(timebut, SIGNAL(clicked()), this, SLOT(changetimemode()) );
  timebut->setPalette(QPalette(QColor(240, 224, 160), QColor(0, 0 , 250)));  
  QToolTip::add(timebut, "Time");

  /*
  slider = new QSlider(Horizontal, this, "Schieber");
  slider->resize(210,10);
  slider->move(40,90);
  slider->setRange(0,99);
  */
  // QPushButton *nameuit = new QPushButton( "Quddit", this, "equit" );

  /*  QPushButton *quit = new QPushButton( "Quit", this, "quit" );
    quit->setFont( QFont( "Times", 18, QFont::Bold ) );

    connect( quit, SIGNAL(clicked()), qApp, SLOT(quit()) );
  */
  lcd  = new QLCDNumber( 2, this, "lcd" );
  lcd->resize(lcdsizey, lcdsizey);
  lcd->move(0,0);
  lcd->setLineWidth(0);
  // lcd->setBackgroundColor(QColor(0,0,0));
  lcd->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));
  lcd->setSegmentStyle(QLCDNumber::Filled);
  QToolTip::add(lcd, "WAVplayer\n(c) 2000 by Michael Me\nhttp://www.michaelmess.de");

  mlcd  = new QLCDNumber( 2, this, "mlcd" );
  mlcd->resize(lcdsizey*9/10, lcdsizey*9/10);
  mlcd->move(lcdsizey, lcdsizey/10);
  mlcd->setLineWidth(0);
  mlcd->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));
  mlcd->setSegmentStyle(QLCDNumber::Filled);
  //  connect(mlcd, SIGNAL(mousePressEvent()), this, SLOT(changetimemode()) );

  slcd  = new QLCDNumber( 2, this, "slcd" );
  slcd->resize(lcdsizey*9/10, lcdsizey*9/10);
  slcd->move(lcdsizey*2,  lcdsizey/10);
  slcd->setLineWidth(0);
  slcd->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));
  slcd->setSegmentStyle(QLCDNumber::Filled);
  // connect(slcd, SIGNAL(clicked()), this, SLOT(changetimemode()) );

  mlabel = new QLabel("M", this, "Minlable", 0);
  mlabel->setFont( QFont( "Times", 16, QFont::Bold ) );
  mlabel->setAutoResize(TRUE);
  mlabel->move(lcdsizey*37/20, lcdsizey*7/10);
  mlabel->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));

  slabel = new QLabel("S", this, "Minlable", 0);
  slabel->setFont( QFont( "Times", 16, QFont::Bold ) );
  slabel->setAutoResize(TRUE);
  slabel->move(lcdsizey*57/20, lcdsizey*7/10);
  slabel->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));

  playlabel = new QLabel("    ", this, "Minlable", 0);
  playlabel->setFont( QFont( "Arial", 18, QFont::Bold ) );
  playlabel->setAutoResize(TRUE);
  playlabel->move(10, lcdsizey);
  playlabel->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));

  timelabel = new QLabel("    ", this, "Minlable", 0);
  timelabel->setFont( QFont( "Arial", 14, QFont::Bold ) );
  timelabel->setAutoResize(TRUE);
  timelabel->move(100, lcdsizey);
  timelabel->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));

  this->setPalette(QPalette(QColor(25, 0, 0), QColor(0, 0 , 126)));
  this->resize(300, 200);

  timer = new QTimer(this);
  connect( timer, SIGNAL(timeout()), SLOT(timeout()));
  timer->start(50);  // Alle 90 millisekunden


  //connect( slider, SIGNAL(valueChanged(int)), lcd, SLOT(display(int)) );
  //connect( trueck, SIGNAL(clicked()), lcd, SLOT(display(88)) );


    /*    QSlider * slider = new QSlider( Horizontal, this, "slider" );
    slider->setRange( 0, 99 );
    slider->setValue( 0 );
    
    connect( slider, SIGNAL(valueChanged(int)), lcd, SLOT(display(int)) );
    */
}



int main( int argc, char **argv )
{
  QApplication app( argc, argv );
  
  MyWidget widget;

  /*
  QPushButton svor( ">>", 0 );
  svor.resize( 100, 30 );
  QPushButton srueck( "<<", 0 );
  srueck.resize( 100, 30 );
  QPushButton tvor( ">>|", 0 );
  tvor.resize( 100, 30 );
  QPushButton trueck( "|<<", 0 );
  trueck.resize( 100, 30 );
  */

  app.setMainWidget( &widget );
  widget.commandlineparameter(argc, argv);
  widget.show();

  return app.exec();
}

#include "wavplayer.moc"








